<?php
/**
 * Domains.lk API Handler
 * 
 * This file handles API requests from the frontend and routes them to the appropriate methods
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/DomainsLkAPI.php';

/**
 * Send JSON response
 */
function sendResponse($success, $data = null, $error = null, $html = null) {
    $response = [
        'success' => $success
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    if ($error !== null) {
        $response['error'] = $error;
    }
    
    if ($html !== null) {
        $response['html'] = $html;
    }
    
    echo json_encode($response);
    exit;
}

try {
    // Get request method
    $method = $_SERVER['REQUEST_METHOD'];
    
    // Get action from POST data
    $action = isset($_POST['action']) ? $_POST['action'] : (isset($_GET['action']) ? $_GET['action'] : 'search');
    
    // Initialize API
    $api = new DomainsLkAPI();
    
    // Route based on action
    switch ($action) {
        case 'search':
        case 'searchDomain':
            // Domain Search
            if (!isset($_POST['domainname']) && !isset($_GET['domainname'])) {
                sendResponse(false, null, 'Domain name is required');
            }
            
            $domainName = isset($_POST['domainname']) ? $_POST['domainname'] : $_GET['domainname'];
            $result = $api->searchDomain($domainName);
            
            // Generate HTML response for domain search results
            $html = generateDomainSearchHTML($result['data'], $domainName);
            
            if ($result['http_code'] == 200 && isset($result['data'])) {
                sendResponse(true, $result['data'], null, $html);
            } else {
                $errorMsg = is_string($result['data']) ? $result['data'] : 'Domain search failed';
                sendResponse(false, $result['data'], $errorMsg);
            }
            break;
            
        case 'provision':
        case 'submitOrder':
            // Submit Provisional Order
            if (!isset($_POST['submission_data'])) {
                sendResponse(false, null, 'Submission data is required');
            }
            
            $submissionData = json_decode($_POST['submission_data'], true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                sendResponse(false, null, 'Invalid JSON in submission data');
            }
            
            $result = $api->submitProvisionalOrder($submissionData);
            
            if ($result['http_code'] == 200) {
                $responseText = is_string($result['data']) ? $result['data'] : json_encode($result['data']);
                if (stripos($responseText, 'success') !== false) {
                    sendResponse(true, $result['data'], null);
                } else {
                    sendResponse(false, $result['data'], $responseText);
                }
            } else {
                sendResponse(false, $result['data'], 'Failed to submit order');
            }
            break;
            
        case 'uploadDocument':
            // Document Upload
            if (!isset($_FILES['file']) || !isset($_POST['domainname']) || !isset($_POST['document_type'])) {
                sendResponse(false, null, 'File, domain name, and document type are required');
            }
            
            $file = $_FILES['file'];
            $domainName = $_POST['domainname'];
            $documentType = $_POST['document_type'];
            
            // Validate file upload
            if ($file['error'] !== UPLOAD_ERR_OK) {
                sendResponse(false, null, 'File upload error: ' . $file['error']);
            }
            
            // Move uploaded file to temp location
            $tempFile = sys_get_temp_dir() . '/' . uniqid() . '_' . $file['name'];
            if (!move_uploaded_file($file['tmp_name'], $tempFile)) {
                sendResponse(false, null, 'Failed to process uploaded file');
            }
            
            try {
                $result = $api->uploadDocument($domainName, $tempFile, $documentType);
                
                // Clean up temp file
                @unlink($tempFile);
                
                if ($result['http_code'] == 200) {
                    $responseText = is_string($result['data']) ? $result['data'] : json_encode($result['data']);
                    if (stripos($responseText, 'success') !== false) {
                        sendResponse(true, $result['data'], null);
                    } else {
                        sendResponse(false, $result['data'], $responseText);
                    }
                } else {
                    sendResponse(false, $result['data'], 'Failed to upload document');
                }
            } catch (Exception $e) {
                @unlink($tempFile);
                // Get the error message (already user-friendly from DomainsLkAPI)
                $errorMessage = $e->getMessage();
                
                // Additional safety: Check if message contains technical details
                if (preg_match('/api\.domains\.lk|cURL Error|Failed to connect|port \d+|Connection timed out/i', $errorMessage)) {
                    if (preg_match('/timeout|connect|timed out/i', $errorMessage)) {
                        $errorMessage = 'Connection timeout. The domain registry service is currently unavailable. Please try again in a few moments.';
                    } else {
                        $errorMessage = 'Unable to connect to the domain registry service. Please try again later or contact support if the problem persists.';
                    }
                }
                
                sendResponse(false, null, $errorMessage);
            }
            break;
            
        case 'suspend':
        case 'unsuspend':
            // Suspend/UnSuspend Domain
            if (!isset($_POST['domainname'])) {
                sendResponse(false, null, 'Domain name is required');
            }
            
            $domainName = $_POST['domainname'];
            $action = ($action === 'suspend') ? 'SuspendRequest' : 'UnSuspendRequest';
            $comment = isset($_POST['comment']) ? $_POST['comment'] : '';
            
            $result = $api->suspendUnsuspendDomain($domainName, $action, $comment);
            
            if ($result['http_code'] == 200) {
                $responseText = is_string($result['data']) ? $result['data'] : json_encode($result['data']);
                if (stripos($responseText, 'success') !== false) {
                    sendResponse(true, $result['data'], null);
                } else {
                    sendResponse(false, $result['data'], $responseText);
                }
            } else {
                sendResponse(false, $result['data'], 'Failed to process request');
            }
            break;
            
        case 'viewDNS':
            // View DNS Records
            if (!isset($_POST['domainname']) && !isset($_GET['domainname'])) {
                sendResponse(false, null, 'Domain name is required');
            }
            
            $domainName = isset($_POST['domainname']) ? $_POST['domainname'] : $_GET['domainname'];
            $result = $api->viewDNSRecords($domainName);
            
            if ($result['http_code'] == 200 && isset($result['data'])) {
                sendResponse(true, $result['data'], null);
            } else {
                sendResponse(false, $result['data'], 'Failed to retrieve DNS records');
            }
            break;
            
        case 'editDNS':
            // Edit DNS Records
            if (!isset($_POST['domainname']) || !isset($_POST['updated_records'])) {
                sendResponse(false, null, 'Domain name and updated records are required');
            }
            
            $domainName = $_POST['domainname'];
            $updatedRecords = json_decode($_POST['updated_records'], true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                sendResponse(false, null, 'Invalid JSON in updated records');
            }
            
            $result = $api->editDNSRecords($domainName, $updatedRecords);
            
            if ($result['http_code'] == 200) {
                $responseText = is_string($result['data']) ? $result['data'] : json_encode($result['data']);
                if (stripos($responseText, 'success') !== false) {
                    sendResponse(true, $result['data'], null);
                } else {
                    sendResponse(false, $result['data'], $responseText);
                }
            } else {
                sendResponse(false, $result['data'], 'Failed to update DNS records');
            }
            break;
            
        default:
            sendResponse(false, null, 'Invalid action: ' . $action);
    }
    
} catch (Exception $e) {
    // Get the error message
    $errorMessage = $e->getMessage();
    
    // Additional safety: Check if message contains technical details and sanitize
    $technicalPatterns = [
        '/api\.domains\.lk/i',
        '/cURL Error/i',
        '/Failed to connect to/i',
        '/port \d+/i',
        '/Connection timed out/i',
        '/CURLE_/i'
    ];
    
    // If message contains technical details, use generic message
    foreach ($technicalPatterns as $pattern) {
        if (preg_match($pattern, $errorMessage)) {
            // Check if it's a timeout/connection issue
            if (preg_match('/timeout|connect|timed out/i', $errorMessage)) {
                $errorMessage = 'Connection timeout. Please try again.';
            } else {
                $errorMessage = 'Unable to connect to the domain registry service. Please try again.';
            }
            break;
        }
    }
    
    sendResponse(false, null, $errorMessage);
}

/**
 * Generate HTML for domain search results
 * This maintains the existing theme and UI structure
 */
function generateDomainSearchHTML($data, $searchedDomain) {
    if (!$data || !is_array($data)) {
        return '<div class="alert alert-danger">No results found or invalid response.</div>';
    }
    
    // Check if domain is available
    $isAvailable = isset($data['Required']['Available']) && $data['Required']['Available'] == 1;
    $message = isset($data['Message']) ? $data['Message'] : '';
    $domainNameFull = isset($data['DomainName']) ? $data['DomainName'] . '.lk' : $searchedDomain . '.lk';
    $expireDate = isset($data['ExpireDate']) ? $data['ExpireDate'] : '';
    $expireDateRes = $expireDate ? $expireDate : '';
    
    // Determine alert class
    $avClass = $isAvailable ? 'success' : 'danger';
    
    // Build the HTML with old handler structure for width constraints
    $html = '<section class="my-5">';
    $html .= '<div class="container">';
    $html .= '<div class="row">';
    $html .= '<div class="lk-domain-result">';
    $html .= '<div class="lk-domain-availability">';
    $html .= '<div class="container">';
    
    // Success/Error Alert
    $html .= '<div class="alert alert-' . $avClass . ' lk-alert" role="alert">';
    $html .= '<div>';
    $html .= '<strong>' . htmlspecialchars($domainNameFull) . ' ' . htmlspecialchars($message);
    if ($expireDateRes) {
        $html .= ' <br> ' . htmlspecialchars($expireDateRes);
    }
    $html .= '</strong>';
    $html .= '</div>';
    $html .= '</div>';
    
    if ($isAvailable) {
        $html .= '<div class="row lkdwhg-row align-items-center justify-content-center">';
        
        // Required Domain (Primary option)
        if (isset($data['Required'])) {
            $required = $data['Required'];
            $domainName = isset($required['DomainName']) ? $required['DomainName'] : $searchedDomain;
            $domainType = isset($required['DomainTypeName']) ? $required['DomainTypeName'] : 'lk';
            $category = isset($required['Category']['Name']) ? $required['Category']['Name'] : '';
            $product = isset($required['Product']) ? $required['Product'] : null;
            $price = isset($product['Price']) ? $product['Price'] : '0.00';
            
            $html .= generateDomainCard($domainName . '.' . $domainType, $category, $price, $product);
        }
        
        // Full Package (if available)
        if (isset($data['FullPackage'])) {
            $fullPackage = $data['FullPackage'];
            $domainName = isset($fullPackage['DomainName']) ? $fullPackage['DomainName'] : $searchedDomain;
            $domainType = isset($fullPackage['DomainTypeName']) ? $fullPackage['DomainTypeName'] : 'lk';
            $category = isset($fullPackage['Category']['Name']) ? $fullPackage['Category']['Name'] : '';
            $product = isset($fullPackage['Product']) ? $fullPackage['Product'] : null;
            $price = isset($product['Price']) ? $product['Price'] : '0.00';
            
            $html .= generateDomainCard($domainName . '.' . $domainType, $category, $price, $product, true);
        }
        
        // Additional Available domains (CAT3, etc.) - Show before Suggested Top Level Domains
        if (isset($data['Additional']) && is_array($data['Additional']) && count($data['Additional']) > 0) {
            $hasAdditionalAvailable = false;
            foreach ($data['Additional'] as $additional) {
                if (isset($additional['Available']) && $additional['Available'] == 1) {
                    $hasAdditionalAvailable = true;
                    break;
                }
            }
            
            if ($hasAdditionalAvailable) {
                $html .= '<h3 class="mt-4 mb-4 text-center">Other options related to your domain search</h3>';
                $html .= '<div class="col-lg-12 col-md-12 col-sm-12 col-12">';
                $html .= '<div class="card p-5">';
                $html .= '<div class="xdffka">';
                $html .= '<div class="xcmkfors"><div class="lk-logo"><img class="img-fluid" src="./assets/img/icons/assets/extentions/lk.png" alt=""></div></div>';
                $html .= '<div class="xckdors">CAT3 - Second Level Domain only</div>';
                $html .= '</div>';
                $html .= '<div class="card-body">';
                $html .= '<ul class="dse95h">';
                
                foreach ($data['Additional'] as $additional) {
                    if (isset($additional['Available']) && $additional['Available'] == 1) {
                        $addDomain = isset($additional['DomainName']) ? $additional['DomainName'] : '';
                        $addType = isset($additional['DomainTypeName']) ? $additional['DomainTypeName'] : 'lk';
                        $addProduct = isset($additional['Product']) ? $additional['Product'] : null;
                        $addPrice = isset($addProduct['Price']) ? $addProduct['Price'] : '0.00';
                        
                        $html .= '<li>';
                        $html .= '<span class="_dsarto">' . htmlspecialchars($addDomain . '.' . $addType) . '</span>';
                        $html .= '<span class="_dsaato">LKR ' . number_format($addPrice, 2) . '/Yr</span>';
                        $html .= '<span><button class="sxfduti" onclick="addToCart(\'' . htmlspecialchars($addDomain . '.' . $addType) . '\')">Buy Now</button></span>';
                        $html .= '</li>';
                    }
                }
                
                $html .= '</ul>';
                $html .= '</div>';
                $html .= '</div>';
                $html .= '</div>';
            }
        }
        
        // Also Available domains (Suggested Top Level Domains) - Show after CAT3
        if (isset($data['AlsoAvailable']) && is_array($data['AlsoAvailable']) && count($data['AlsoAvailable']) > 0) {
            $html .= '<div class="col-lg-12 mt-4 col-md-12 col-sm-12 col-12">';
            $html .= '<div class="card p-5">';
            $html .= '<div class="xdffka">';
            $html .= '<div class="xcmkfors"><div class="lk-logo"><img class="img-fluid" src="./assets/img/icons/assets/extentions/lk.png" alt=""></div></div>';
            $html .= '<div class="xckdors">Suggested Top Level Domains</div>';
            $html .= '</div>';
            $html .= '<div class="card-body">';
            $html .= '<ul class="dse95h">';
            
            foreach ($data['AlsoAvailable'] as $also) {
                $alsoDomain = isset($also['DomainName']) ? $also['DomainName'] : '';
                $alsoType = isset($also['DomainTypeName']) ? $also['DomainTypeName'] : 'lk';
                $alsoProduct = isset($also['Product']) ? $also['Product'] : null;
                $alsoPrice = isset($alsoProduct['Price']) ? $alsoProduct['Price'] : '0.00';
                
                $html .= '<li>';
                $html .= '<span class="_dsarto">' . htmlspecialchars($alsoDomain . '.' . $alsoType) . '</span>';
                $html .= '<span class="_dsaato">LKR ' . number_format($alsoPrice, 2) . '/Yr</span>';
                $html .= '<span><button class="sxfduti" onclick="addToCart(\'' . htmlspecialchars($alsoDomain . '.' . $alsoType) . '\')">Buy Now</button></span>';
                $html .= '</li>';
            }
            
            $html .= '</ul>';
            $html .= '</div>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        $html .= '</div>'; // Close row lkdwhg-row
    }
    
    // Close nested containers (matching old handler structure)
    $html .= '</div>'; // Close inner container
    $html .= '</div>'; // Close lk-domain-availability
    $html .= '</div>'; // Close lk-domain-result
    $html .= '</div>'; // Close row
    $html .= '</div>'; // Close outer container
    $html .= '</section>';
    
    return $html;
}

/**
 * Generate domain card HTML
 */
function generateDomainCard($domainName, $category, $price, $product, $isFullPackage = false) {
    $colClass = $isFullPackage ? 'col-lg-6 col-md-6 col-sm-12 col-12 mb-4' : 'col-lg-6 col-md-6 col-sm-12 col-12 mb-4';
    $cardClass = $isFullPackage ? 'card p-5 break-word' : 'card p-5';
    
    $html = '<div class="' . $colClass . '">';
    $html .= '<div class="' . $cardClass . '">';
    $html .= '<div class="xdffka">';
    $html .= '<div class="xcmkfors"><div class="lk-logo"><img class="img-fluid" src="./assets/img/icons/assets/extentions/lk.png" alt=""></div></div>';
    $html .= '<div class="xckdors">' . htmlspecialchars($category) . '</div>';
    $html .= '</div>';
    $html .= '<div class="card-body' . ($isFullPackage ? ' break-word' : '') . '">';
    $html .= '<div class="umbdse">';
    $html .= '<p><strong>*' . htmlspecialchars($domainName) . '</strong></p>';
    $html .= '<span>LKR ' . number_format($price, 2) . '<small>/Yr</small></span>';
    $html .= '</div>';
    $html .= '<p class="mt-3">' . htmlspecialchars($category) . '</p>';
    
    if ($isFullPackage) {
        $html .= '<strong>*You automatically reserve following names</strong>';
        $html .= '<ul class="umnbkor">';
        $baseName = preg_replace('/\.lk$/', '', $domainName);
        $html .= '<li>' . htmlspecialchars($baseName . '.edu.lk') . '</li>';
        $html .= '<li>' . htmlspecialchars($baseName . '.com.lk') . '</li>';
        $html .= '<li>' . htmlspecialchars($baseName . '.hotel.lk') . '</li>';
        $html .= '<li>' . htmlspecialchars($baseName . '.org.lk') . '</li>';
        $html .= '<li>' . htmlspecialchars($baseName . '.web.lk') . '</li>';
        $html .= '</ul>';
    } else {
        $html .= '<strong>Here you will get a Top Level Domain Only. We recommend that you get the Full Domain Package related to this domain.</strong>';
    }
    
    $html .= '<p class="' . ($isFullPackage ? 'mt-3' : 'xomgks') . '">';
    $html .= '<button class="sxfduti" onclick="addToCart(\'' . htmlspecialchars($domainName) . '\')">Buy Now</button>';
    $html .= '</p>';
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    
    return $html;
}