<?php
/**
 * Domains.lk API Handler Class
 * 
 * This class provides a clean interface to interact with the domains.lk API
 * All methods follow standard PHP practices and are designed for maintainability
 */

require_once __DIR__ . '/config.php';

class DomainsLkAPI {
    
    private $apiKey;
    private $baseUrl;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->apiKey = DOMAINS_LK_API_KEY;
        $this->baseUrl = DOMAINS_LK_API_BASE_URL;
        
        if (!$this->apiKey) {
            throw new Exception('API key not configured. Please update config.php with your actual API key.');
        }
    }
    
    /**
     * Make a cURL request to the API
     * 
     * @param string $url Full API URL
     * @param string $method HTTP method (GET, POST, etc.)
     * @param array $headers HTTP headers
     * @param mixed $data Request data (array for JSON, string for form data)
     * @param bool $isJson Whether to send data as JSON
     * @return array Response data
     * @throws Exception
     */
    private function makeRequest($url, $method = 'POST', $headers = [], $data = null, $isJson = true) {
        $ch = curl_init($url);
        
        // Default headers
        $defaultHeaders = [
            'x-api-key: ' . $this->apiKey
        ];
        
        // Merge headers
        $allHeaders = array_merge($defaultHeaders, $headers);
        
        // Set cURL options
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $allHeaders);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        
        // Add data if provided
        if ($data !== null) {
            if ($isJson) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            } else {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            }
        }
        
        // Execute request
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        // Handle cURL errors
        if ($error) {
            throw new Exception('cURL Error: ' . $error);
        }
        
        // Decode JSON response
        $decoded = json_decode($response, true);
        
        // Return response
        return [
            'http_code' => $httpCode,
            'data' => $decoded !== null ? $decoded : $response,
            'raw' => $response
        ];
    }
    
    /**
     * Search for domain availability
     * 
     * @param string $domainName Domain name to search (with or without .lk)
     * @return array API response
     */
    public function searchDomain($domainName) {
        // Clean domain name
        $domainName = trim($domainName);
        
        // Remove .lk if present (API expects just the name)
        $domainName = preg_replace('/\.lk$/i', '', $domainName);
        
        if (empty($domainName)) {
            throw new Exception('Domain name is required');
        }
        
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_SEARCH;
        $data = [
            'domainname' => $domainName
        ];
        
        $headers = [
            'Content-Type: application/json'
        ];
        
        return $this->makeRequest($url, 'POST', $headers, $data, true);
    }
    
    /**
     * Submit provisional order for domain registration
     * 
     * @param array $submissionData Submission data array
     * @return array API response
     */
    public function submitProvisionalOrder($submissionData) {
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_PROVISION;
        
        // Prepare form data
        $data = [
            'SubmissionInformation' => json_encode($submissionData)
        ];
        
        $headers = [];
        
        return $this->makeRequest($url, 'POST', $headers, $data, false);
    }
    
    /**
     * Upload document for domain registration
     * 
     * @param string $domainName Domain name
     * @param string $filePath Path to the file to upload
     * @param string $documentType Document type
     * @return array API response
     */
    public function uploadDocument($domainName, $filePath, $documentType) {
        if (!file_exists($filePath)) {
            throw new Exception('File not found: ' . $filePath);
        }
        
        // Validate file size
        $fileSize = filesize($filePath);
        if ($fileSize > DOMAINS_LK_MAX_FILE_SIZE) {
            throw new Exception(DOMAINS_LK_ERROR_FILE_TOO_LARGE);
        }
        
        // Validate file extension
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
        if (!in_array($extension, DOMAINS_LK_ALLOWED_EXTENSIONS)) {
            throw new Exception(DOMAINS_LK_ERROR_INVALID_FILE_TYPE);
        }
        
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_API;
        
        // Prepare multipart form data
        $data = [
            'Action' => 'DocumentUpload',
            'DomainName' => $domainName,
            'DocumentType' => $documentType,
            'File' => new CURLFile($filePath)
        ];
        
        $headers = [];
        
        return $this->makeRequest($url, 'POST', $headers, $data, false);
    }
    
    /**
     * Suspend or unsuspend a domain
     * 
     * @param string $domainName Domain name
     * @param string $action 'SuspendRequest' or 'UnSuspendRequest'
     * @param string $comment Optional comment
     * @return array API response
     */
    public function suspendUnsuspendDomain($domainName, $action, $comment = '') {
        if (!in_array($action, ['SuspendRequest', 'UnSuspendRequest'])) {
            throw new Exception('Invalid action. Must be SuspendRequest or UnSuspendRequest');
        }
        
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_API;
        
        $data = [
            'Action' => $action,
            'DomainName' => $domainName,
            'Comment' => $comment
        ];
        
        $headers = [];
        
        return $this->makeRequest($url, 'POST', $headers, $data, false);
    }
    
    /**
     * View DNS records for a domain
     * 
     * @param string $domainName Domain name
     * @return array API response
     */
    public function viewDNSRecords($domainName) {
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_API;
        
        $data = [
            'Action' => 'ViewDNSRecords',
            'DomainName' => $domainName
        ];
        
        $headers = [];
        
        return $this->makeRequest($url, 'POST', $headers, $data, false);
    }
    
    /**
     * Edit DNS records for a domain
     * 
     * @param string $domainName Domain name
     * @param array $updatedRecords Array with NameServers or ResourceRecords
     * @return array API response
     */
    public function editDNSRecords($domainName, $updatedRecords) {
        // Validate records count
        $totalRecords = 0;
        if (isset($updatedRecords['NameServers'])) {
            $totalRecords += count($updatedRecords['NameServers']);
        }
        if (isset($updatedRecords['ResourceRecords'])) {
            $totalRecords += count($updatedRecords['ResourceRecords']);
        }
        
        if ($totalRecords > DOMAINS_LK_MAX_DNS_RECORDS) {
            throw new Exception('Maximum ' . DOMAINS_LK_MAX_DNS_RECORDS . ' records allowed');
        }
        
        // Cannot have both NameServers and ResourceRecords
        if (isset($updatedRecords['NameServers']) && isset($updatedRecords['ResourceRecords'])) {
            throw new Exception('Cannot submit both NameServers and ResourceRecords in a single submission');
        }
        
        $url = $this->baseUrl . DOMAINS_LK_ENDPOINT_API;
        
        $data = [
            'Action' => 'EditDNSRecords',
            'DomainName' => $domainName,
            'UpdatedRecords' => json_encode($updatedRecords)
        ];
        
        $headers = [];
        
        return $this->makeRequest($url, 'POST', $headers, $data, false);
    }
}

